/*
 * @(#)Diagram.java  1.0  2. Dezember 2003
 *
 * Copyright (c) 2003 Lucerne University of Applied Sciences and Arts (HSLU)
 * Zentralstrasse 18, Postfach 2858, CH-6002 Lucerne, Switzerland
 * All rights reserved.
 *
 * The copyright of this software is owned by the Lucerne University of Applied 
 * Sciences and Arts (HSLU). You may not use, copy or modify this software, 
 * except in accordance with the license agreement you entered into with HSLU. 
 * For details see accompanying license terms. 
 */

package ch.hslu.cm;

import ch.hslu.cm.simulation.*;
import java.io.IOException;
import org.jhotdraw.draw.DefaultDrawing;
import org.jhotdraw.draw.Figure;
import org.jhotdraw.xml.DOMInput;
import org.jhotdraw.xml.DOMOutput;
import java.util.*;
/**
 * A diagram is a drawing which visualizes a simulation.
 *
 * @author  Werner Randelshofer
 * @version 1.0 2. Dezember 2003  Created.
 */
public class AbstractDiagram
extends DefaultDrawing
//extends QuadTreeDrawing
implements Diagram, SimulationListener {
    private Simulation simulation;
    
    /**
     * This map contains two entries for each SimulationElement which has
     * a DiagramFigure.
     * The first entry maps from the SimulationElement to the DiagramFigure.
     * The second entry maps from the DiagramFigure to the SimulationElement.
     */
    private HashMap<Object,Object> simulationMap = new HashMap<Object,Object>();
    
    /** Creates a new instance. */
    public AbstractDiagram() {
    }
    
    public Simulation getSimulation() {
        return simulation;
    }
    
    /**
     * Sets the simulation and creates a diagram for the simulation.
     */
    public void setSimulation(Simulation simulation) {
        if (this.simulation != null) {
            this.simulation.removeSimulationListener(this);
            removeAllChildren();
            simulationMap.clear();
        }
        this.simulation = simulation;
        if (this.simulation != null) {
            this.simulation.addSimulationListener(this);
            createDiagram();
        }
    }
    /**
     * Sets the simulation without changing the contents of the diagram.
     */
    public void putSimulation(Simulation simulation) {
        if (this.simulation != null) {
            this.simulation.removeSimulationListener(this);
        }
        this.simulation = simulation;
        if (this.simulation != null) {
            this.simulation.addSimulationListener(this);
            createMap();
        }
    }
    
    public void basicAdd(int index, Figure figure) {
        if (figure instanceof DiagramFigure) {
            map(((DiagramFigure) figure).getModel(), (DiagramFigure) figure);
        }
        super.basicAdd(index, figure);
    }
    public int basicRemove(Figure figure) {
        if (figure instanceof DiagramFigure) {
            unmap(((DiagramFigure) figure).getModel(), (DiagramFigure) figure);
        }
        return super.basicRemove(figure);
    }
    
    
    protected void map(SimulatedObject simElem, DiagramFigure diagElem) {
        simulationMap.put(diagElem, simElem);
        simulationMap.put(simElem, diagElem);
    }
    protected void unmap(SimulatedObject simElem, DiagramFigure diagElem) {
        simulationMap.remove(diagElem);
        simulationMap.remove(simElem);
    }
    protected DiagramFigure getDiagramFigure(SimulatedObject simElem) {
        return (DiagramFigure) simulationMap.get(simElem);
    }
    protected SimulatedElement getElement(DiagramFigure diagElem) {
        return (SimulatedElement) simulationMap.get(diagElem);
    }
    
    
    /**
     * Creates the diagram from the simulation.
     */
    protected void createDiagram() {
    }
    /**
     * Updates the simulationMap
     */
    protected void createMap() {
        simulationMap.clear();
        
        // Create a decomposition of all figures in the diagram.
        LinkedList<Figure> decomposition = new LinkedList<Figure>();
        LinkedList<Figure> composites = new LinkedList<Figure>(getChildren());
        while (composites.size() > 0) {
            Collection figures = composites;
            composites = new LinkedList<Figure>();
            for (Iterator i=figures.iterator(); i.hasNext(); ) {
                Figure f = (Figure) i.next();
                LinkedList<Figure> fd = new LinkedList<Figure>(f.getDecomposition());
                decomposition.add(f);
                fd.remove(f);
                composites.addAll(fd);
            }
        }
        
        // Create the maps of model elements and figures.
        for (Iterator i=decomposition.iterator(); i.hasNext(); ) {
            Figure f = (Figure) i.next();
            if (f instanceof DiagramFigure) {
                DiagramFigure df = (DiagramFigure) f;
                simulationMap.put(df, df.getModel());
                simulationMap.put(df.getModel(), df);
            }
        }
    }
    
    /**
     * Creates a diagram element for the given simulated element.
     * Not each simulated element must have a diagram element. This method
     * returns null, if there is no diagram element which can visualize
     * the simulated element.
     */
    protected DiagramFigure createDiagramFigure(SimulatedObject simElem) {
        return null;
    }
    
    public void behaviourFinished(SimulationEvent evt) {
    }
    
    public void behaviourStarted(SimulationEvent evt) {
    }
    
    public void objectAdded(SimulationEvent evt) {
        if (getDiagramFigure(evt.getObject()) == null) {
            DiagramFigure diagramElement = createDiagramFigure(evt.getObject());
            if (diagramElement != null) {
                synchronized(getLock()) {
                    add(diagramElement);
                }
            }
        }
    }
    
    public void objectRemoved(SimulationEvent evt) {
    }
    
    @Override
    public void read(DOMInput in) throws IOException {
        in.openElement((in.getElementCount("model") == 1) ? "model" : "Model");
        setSimulation((Simulation) in.readObject(0));
        in.closeElement();
        super.read(in);
    }
    @Override
    public void write(DOMOutput out) throws IOException {
        out.openElement("Model");
        out.writeObject(getSimulation());
        out.closeElement();
        super.write(out);
    }
    
    public DiagramFigure findFigure(SimulatedElement element) {
        for (Figure f : getChildren()) {
            if (f instanceof DiagramFigure) {
                DiagramFigure df = (DiagramFigure) f;
                if (df.getModel() == element) {
                    return df;
                }
            }
        }
        return null;
    }
    
}
